///////////////////////////////////////////////////////////////////////////////
///  Table-driven LL(1) Parser
///
///  (c) Michael L. Scott, 2023-2025
///  For use by students in CSC 2/454 at the University of Rochester
///  during the Fall 2025 term.  All other use requires written
///  permission of the author.
///
///  Uses tables, in module 'tables', generated by separate table_gen tool.
///  Predicts epsilon productions using global FOLLOW sets.
///
//  *** Currently does no error recovery: panics on syntax errors.
//  *** You need to fix that.
//
//  *** Space for attributes is maintained automatically using the
//  *** mechanism described in Section 4.9 on the PLP5e companion site.
//  *** You should disable this mechanism after recovering from a syntax error.
//

use crate::tables::Tkn;
use Tkn::*;
use crate::tables::{Goal, PARSE_TAB, PROD_TAB, PSitem};
// *** You'll also want to import FIRST and FOLLOW to do error recovery.
use crate::tables::Act::*;
use PSitem::*;
use crate::scanner::Scanner;
use crate::scanner::Token;
use crate::attributes::*;
use crate::actions::do_action;

use std::cell::Cell;

// *** Anything that needs to be shared among methods in impl Parser needs to
// *** be a field of this struct.
pub struct Parser {
  scanner: Scanner,
  next_tkn: Token,          // already peeked at
  parse_stack: Vec<PSitem>,
}

impl Parser {
  pub fn new() -> Self {
    Self {
      scanner: Scanner::new(),
      next_tkn: Token { tp: TknSIZE, text: String::new(), line: 0, col: 0 },
      parse_stack: Vec::new(),
    }
  }

  // I'd call this "match", but that's a keyword.
  // Return value indicates whether we ate a real token
  fn eat(&mut self, expected: Tkn) -> bool {
    if self.next_tkn.tp == expected {
      println!("match {:?}: {}", expected, self.next_tkn.text);
      self.next_tkn = self.scanner.scan();
      return true;
    }
    return false;
  }

  // main entry point
  pub fn parse(&mut self) {
    // prime the pump:
    self.next_tkn = self.scanner.scan();
    self.parse_stack.push(NT(Goal));

    let mut attr_stack: Vec<Cell<ASitem>> = Vec::new();
      // As it turns out, it's difficult in Rust to take ownership of an item
      // in a vector (without actually removing it and sliding everything else
      // down at O(n) cost).  We make it easier by letting each item be a
      // Cell<ASitem> object, which supports ownership transferring .take()
      // and .set(x) methods.  The .take() method requires that the ASitem type
      // have a default value (Null), which is put into the vector instead.  The
      // .set(x) method drops the previous content.
    attr_stack.push(Cell::new(Null));   // Goal
    let mut l_attr: usize = 0;    // \
    let mut r_attr: usize = 0;    //   > indices into attr_stack
    let mut n_attr: usize = 0;    // /
    let mut old_contexts: Vec<(usize, usize)> = Vec::new();
      // old (l_attr, r_attr) pairs

    loop {
      match self.parse_stack.pop() {
        None => panic!("parse stack underflow; there's a bug somewhere"),
        Some(Tk(tok)) => {
          // put token we just popped into the attribute stack:
          attr_stack[n_attr].set(Tok(self.next_tkn.clone()));
          n_attr += 1;
          if self.eat(tok) {
            if tok == Stop {
              // *** print out the constructed AST here
              return;   // end of program
            }
          } else {  // didn't find expected token
            // *** You'll need to recover here instead
            panic!("syntax error on line {}: expected {:?} but saw {:?}",
                   self.next_tkn.line, tok, self.next_tkn.tp);
          }
        }
        Some(NT(lhs)) => {
          let ntk = self.next_tkn.tp;
          let action = &PARSE_TAB[lhs as usize][ntk as usize];
          match action {
            Err => {
              // *** You'll need to recover here instead
              panic!("syntax error on line {}: no prediction for {:?} when seeing {:?}",
                     self.next_tkn.line, lhs, ntk);
            }
            Prod(n) | EProd(n) => {
              let rhs = PROD_TAB[*n];
                // n is a reference.  It can't be implicitly dereferenced
                // because it can come from more than one place.
              println!("predict {} -->{}", lhs,
                rhs.iter().fold(String::new(), |s, i| s + " " + &i.to_string()));
              // save indices in attribute stack:
              old_contexts.push((l_attr, r_attr));
              l_attr = n_attr;
              r_attr = attr_stack.len();
              n_attr = r_attr;
              // mark end of production in the parse stack:
              self.parse_stack.push(EoP);
              // update parse and attribute stacks:
              for &s in rhs.iter().rev() {
                self.parse_stack.push(s);
                attr_stack.push(Cell::new(Null));
              }
              if let EProd(_) = action {
                // *** This is where you would need to do something special if
                // *** trying to avoid the immediate error detection problem.
              }
            }
          }
        }
        Some(EoP) => {  // end of current production
          n_attr = l_attr + 1;
          attr_stack.truncate(r_attr);
          match old_contexts.pop() {
            None => panic!("context stack underflow; there's a bug somewhere"),
            Some(pair) => {
              (l_attr, r_attr) = pair;
            }
          }
        }
        Some(AR(n)) => {
          do_action(n, &mut attr_stack, l_attr, r_attr);
          n_attr += 1;
        }
      }
    }
  }

} // end impl Parser
